from sqlalchemy import Column, Integer, String, Text, ForeignKey, Boolean
from sqlalchemy.orm import relationship
from database import Base

# -------------------- Recruiter Profile --------------------
class RecruiterProfile(Base):
    __tablename__ = "recruiter_profiles"
    id = Column(Integer, primary_key=True, index=True)
    user_id = Column(Integer, ForeignKey("users.id"), unique=True)
    first_name = Column(String(100), nullable=True)
    last_name = Column(String(100), nullable=True)
    introduction = Column(Text, nullable=True)
    website = Column(String(100), nullable=True)
    year_founded = Column(String(100), nullable=True)
    number_of_employees = Column(String(100), nullable=True)
    location = Column(String(100), nullable=True)
    email = Column(String(100), nullable=True)
    company_logo = Column(String(100), nullable=True)  # path to logo file


# -------------------- Student Profile --------------------
class StudentProfile(Base):
    __tablename__ = "student_profiles"
    id = Column(Integer, primary_key=True, index=True)
    user_id = Column(Integer, ForeignKey("users.id"), unique=True)
    first_name = Column(String(100), nullable=False)
    last_name = Column(String(100), nullable=False)
    dob = Column(String(100), nullable=True)
    gender = Column(String(100), nullable=True)
    contact = Column(String(100), nullable=True)
    address = Column(String(100), nullable=True)
    education = Column(String(100), nullable=True)
    email = Column(String(100), nullable=False)
    profile_picture = Column(String(100), nullable=True)  # path to image file
    video_path = Column(String(255), nullable=True)


    # Relationships
    educations = relationship("EducationHistory", backref="student", cascade="all, delete")
    experiences = relationship("WorkExperience", backref="student", cascade="all, delete")
    certifications = relationship("Certification", backref="student", cascade="all, delete")
    skills = relationship("Skill", backref="student", cascade="all, delete")

# -------------------- Recruiter Trial Students --------------------
class RecruiterTrialStudent(Base):
    __tablename__ = "recruiter_trial_students"
    id = Column(Integer, primary_key=True, index=True)
    recruiter_id = Column(Integer, ForeignKey("users.id"))
    student_id = Column(Integer, ForeignKey("student_profiles.id"))



# -------------------- Education History --------------------
class EducationHistory(Base):
    __tablename__ = "education_history"
    id = Column(Integer, primary_key=True, index=True)
    student_id = Column(Integer, ForeignKey("student_profiles.id"))
    level = Column(String(100))  # Matriculation, Intermediate, etc.
    institution_name = Column(String(100))
    percentage_or_cgpa = Column(String(100))
    address = Column(String(100))
    is_pursuing = Column(Boolean, default=False)
    marksheet_path = Column(String(100), nullable=True)


# -------------------- Work Experience --------------------
class WorkExperience(Base):
    __tablename__ = "work_experience"
    id = Column(Integer, primary_key=True, index=True)
    student_id = Column(Integer, ForeignKey("student_profiles.id"))
    job_title = Column(String(100))
    company_name = Column(String(100))
    duration_from = Column(String(100))
    duration_to = Column(String(100))  # Can be 'Present' if still working
    description = Column(Text)


# -------------------- Certifications --------------------
class Certification(Base):
    __tablename__ = "certifications"
    id = Column(Integer, primary_key=True, index=True)
    student_id = Column(Integer, ForeignKey("student_profiles.id"))
    title = Column(String(100))
    organization = Column(String(100))
    certificate_path = Column(String(100))


# -------------------- Admin Profile --------------------
class AdminProfile(Base):
    __tablename__ = "admin_profiles"
    id = Column(Integer, primary_key=True, index=True)
    user_id = Column(Integer, ForeignKey("users.id"), unique=True)
    first_name = Column(String(100), nullable=False)
    last_name = Column(String(100), nullable=False)
    email = Column(String(100), nullable=False)
    contact = Column(String(100), nullable=True)
    designation = Column(String(100), nullable=True)


class Skill(Base):
    __tablename__ = "skills"
    id = Column(Integer, primary_key=True, index=True)
    student_id = Column(Integer, ForeignKey("student_profiles.id"))
    name = Column(String(100), nullable=False)